#				CAWS
#		Crew Advisory and Warning System
#
#		Bea Wolf, started February 2024
#		
#		ref. PIM, Section 7-23

#		Loosely based on CRJ700 EICAS Message System by Ryan Miller


#	Concepts
#
#		Every message / "warning light" is an object
#		The system checks the status of all message conditions
#		If a new message is detected, the system lights the master caution/warning and stores the "causing" message as well as initiates the voice callout
#		Master Caution/Warning is extinguished either by pressing the button or when the message condition becomes false

var message_index = [];

var voltage = [
	props.globals.getNode("/systems/electrical/outputs/caws[0]", 1),
	props.globals.getNode("/systems/electrical/outputs/caws[1]", 1),
];

var master_warning_int = -1;
var master_caution_int = -1;

var master_warning = props.globals.initNode("/instrumentation/caws/master-warning", 0, "BOOL");
var master_caution = props.globals.initNode("/instrumentation/caws/master-caution", 0, "BOOL");

var message = {
	new: func( name, priority, condition ){
		var m = { parents: [ message ] };
		m.name = name;
		m.node = props.globals.initNode("/instrumentation/caws/"~ name, 0, "BOOL");
		m.priority = priority;		# 0 = warning (red), 1 = caution (amber), 2 = advisory (green)
		m.condition = condition;
		m.status = 0;			# 0 = not active, 1 = active
		m.ack = 0;				# 1 = message acknowledged (by pressing the master caution/warning)
		return m;
	},
	update: func{
		if ( props.condition( me.condition ) ) {
			me.status = 1;
			me.node.setBoolValue( 1 );
			if( !me.ack ){
				return me.priority;
			} else {
				return -1;
			}
		} else {
			if( me.status ){
				# message is active but condition has vanished
				# reset
				me.status = 0;
				me.node.setBoolValue( 0 );
				me.ack = 0;
				return -1;
			} else {
				# message is not active and condition is false
				return -1;
			}
		}
	},
};

var main_loop = func{
	if( voltage[0].getDoubleValue() > 21.0 or voltage[1].getDoubleValue() > 21.0 ){
	
		master_warning_int = 0;
		master_caution_int = 0;
		
		foreach( var el; message_index ){
			var exit_status = el.update();
			
			if( exit_status == 0 ){
				# new warning condition
				if( master_warning_int != 1 ){
					master_warning_int = 1;
				}
			} elsif( exit_status == 1 ){
				# new caution condition
				if( master_caution_int != 1 ){
					master_caution_int = 1;
				}
			}
		}
		
		master_warning.setBoolValue( master_warning_int );
		master_caution.setBoolValue( master_caution_int );
		
	}
}

var main_loop_timer = maketimer( 0.5, main_loop );
main_loop_timer.simulatedTime = 1;


var load_messages = func{	
	var messages = io.read_properties("Nasal/caws-messages.xml").getChildren("message");
	foreach (var msg; messages) {
		append(message_index, message.new( msg.getNode("name", 1).getValue(), msg.getNode("priority", 1).getIntValue(), msg.getNode("condition", 1) ) );
	}
}

var master_warning_ack = func{
	foreach( var el; message_index ){
		if( el.status == 1 and el.priority == 0 ){
			el.ack = 1;
		}
	}
}
var master_caution_ack = func{
	foreach( var el; message_index ){
		if( el.status == 1 and el.priority == 1 ){
			el.ack = 1;
		}
	}
}

var startup_listener = setlistener("/sim/signals/fdm-initialized", func{
	load_messages();
	main_loop_timer.start();
	removelistener( startup_listener );
});
